import React, { useState } from 'react';
import { BrowserRouter as Router, Routes, Route } from 'react-router-dom';
import Navbar from './components/Navbar';
import Dashboard from './components/Dashboard';
import RoomCard from './components/RoomCard';
import BookingForm from './components/BookingForm';
import { getRooms } from './utils/dataManager';
import { FaHotel, FaPhone, FaEnvelope, FaMapMarker } from 'react-icons/fa';
import './styles.css';

const Hero = () => (
  <div className="hero">
    <div className="hero-content">
      <h1>Welcome to Luxury Hotel</h1>
      <p>Experience comfort and elegance at its finest</p>
    </div>
  </div>
);

const Footer = () => (
  <footer className="footer">--
    <div className="footer-content">
      <div>
        <h3><FaPhone /> Contact</h3>
        <p>+1 234 567 890</p>
        <p><FaEnvelope /> info@luxuryhotel.com</p>
      </div>
      <div>
        <h3><FaMapMarker /> Location</h3>
        <p>123 Luxury Street</p>
        <p>Paradise City, 12345</p>
      </div>
    </div>
    <div className="footer-bottom">
      <p>&copy; 2024 Luxury Hotel. All rights reserved.</p>
    </div>
  </footer>
);

const App = () => {
  const [bookings, setBookings] = useState([]);
  const [selectedRoom, setSelectedRoom] = useState(null);
  const rooms = getRooms();

  const handleBookRoom = (room) => {
    setSelectedRoom(room);
  };

  const handleBookingSubmit = (bookingData) => {
    setBookings([...bookings, { ...bookingData, id: Date.now(), room: selectedRoom }]);
    setSelectedRoom(null);
  };

  return (
    <Router>
      <div className="app">
        <Navbar />
        <Routes>
          <Route path="/" element={
            <>
              <Hero />
              <Dashboard 
                totalRooms={rooms.length}
                availableRooms={rooms.length - bookings.length}
                bookings={bookings}
              />
            </>
          } />
          <Route path="/rooms" element={
            <div className="room-container">
              <h2 className="page-title"><FaHotel /> Available Rooms</h2>
              <div className="room-list">
                {rooms.map(room => (
                  <RoomCard 
                    key={room.id}
                    room={{
                      ...room,
                      image: room.type === "Single" 
                        ? "https://images.unsplash.com/photo-1631049307264-da0ec9d70304"
                        : room.type === "Double"
                        ? "https://images.unsplash.com/photo-1595576508898-0ad5c879a061"
                        : "https://images.unsplash.com/photo-1590490360182-c33d57733427"
                    }}
                    onBookRoom={handleBookRoom}
                  />
                ))}
              </div>
              {selectedRoom && (
                <BookingForm 
                  room={selectedRoom}
                  onSubmit={handleBookingSubmit}
                  onClose={() => setSelectedRoom(null)}
                />
              )}
            </div>
          } />
          <Route path="/bookings" element={
            <div className="bookings-container">
              <h2 className="page-title">Current Bookings</h2>
              <div className="bookings-list">
                {bookings.map(booking => (
                  <div key={booking.id} className="booking-card">
                    <img 
                      src={booking.room.image} 
                      alt={booking.room.type} 
                      className="booking-image"
                    />
                    <div className="booking-details">
                      <h3>Guest: {booking.name}</h3>
                      <p>Room: {booking.room.type}</p>
                      <p>Check In: {booking.checkIn}</p>
                      <p>Check Out: {booking.checkOut}</p>
                      <p>Guests: {booking.guests}</p>
                      <p>Email: {booking.email}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          } />
        </Routes>
        <Footer />
      </div>
    </Router>
  );
};

export default App;